<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use App\Models\AdPackage;
use App\Models\DisplayZone;
use App\Models\Location;
use Carbon\Carbon;
use Illuminate\Http\Request;

class AdController extends Controller
{
    /**
     * نمایش صفحه مدیریت تبلیغات مکان‌ها
     */
    public function locationAds()
    {
        $ads = Ad::with(['location', 'package'])
            ->where('ad_type', 'location')
            ->orderBy('end_date', 'desc')
            ->paginate(20);

        return view('admin.ads.location-ads', compact('ads'));
    }

    /**
     * نمایش صفحه تخصیص پکیج تبلیغات به مکان
     */
    public function assignPackage()
    {
        $locations = Location::where('status', 'approved')
            ->orderBy('name', 'asc')
            ->get();

        $packages = AdPackage::where('is_active', true)
            ->where('ad_type', 'location')
            ->orderBy('price', 'asc')
            ->get();

        return view('admin.ads.assign-package', compact('locations', 'packages'));
    }

    /**
     * ذخیره تخصیص پکیج تبلیغات به مکان
     */
    public function storeAssignment(Request $request)
    {
        $request->validate([
            'location_id' => 'required|exists:locations,id',
            'package_id' => 'required|exists:ad_packages,id',
            'start_date' => 'required|date|after_or_equal:today',
            'payment_method' => 'required|in:cash,card,free',
            'tracking_code' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);

        // دریافت پکیج برای محاسبه تاریخ پایان
        $package = AdPackage::findOrFail($request->package_id);

        // تبدیل تاریخ شروع از فرمت شمسی به میلادی
        $startDate = Carbon::createFromFormat('Y/m/d', $request->start_date)->startOfDay();
        $endDate = (clone $startDate)->addDays($package->duration_days);

        // ایجاد تبلیغ جدید
        $ad = new Ad;
        $ad->location_id = $request->location_id;
        $ad->package_id = $request->package_id;
        $ad->ad_type = 'location';
        $ad->start_date = $startDate;
        $ad->end_date = $endDate;
        $ad->payment_method = $request->payment_method;
        $ad->tracking_code = $request->tracking_code;
        $ad->notes = $request->notes;
        $ad->status = Carbon::now()->startOfDay()->gte($startDate) ? 'active' : 'pending';
        $ad->created_by = auth()->id();
        $ad->save();

        return redirect()->route('admin.ads.location')
            ->with('success', 'پکیج تبلیغاتی با موفقیت به مکان مورد نظر تخصیص داده شد.');
    }

    /**
     * فعال‌سازی تبلیغ
     */
    public function activateAd(Request $request)
    {
        $ad = Ad::findOrFail($request->ad_id);
        $ad->status = 'active';
        $ad->save();

        return redirect()->back()->with('success', 'تبلیغ با موفقیت فعال شد.');
    }

    /**
     * غیرفعال‌سازی تبلیغ
     */
    public function deactivateAd(Request $request)
    {
        $ad = Ad::findOrFail($request->ad_id);
        $ad->status = 'inactive';
        $ad->save();

        return redirect()->back()->with('success', 'تبلیغ با موفقیت غیرفعال شد.');
    }

    /**
     * تمدید تبلیغ
     */
    public function extendAd(Request $request)
    {
        $request->validate([
            'ad_id' => 'required|exists:ads,id',
            'extend_days' => 'required|integer|min:1|max:365',
            'payment_method' => 'required|in:cash,card,free',
            'tracking_code' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);

        $ad = Ad::findOrFail($request->ad_id);

        // اگر تبلیغ منقضی شده، تاریخ جدید را از امروز شروع می‌کنیم
        if ($ad->end_date < Carbon::now()) {
            $ad->end_date = Carbon::now();
        }

        // افزودن روزهای تمدید به تاریخ پایان
        $ad->end_date = Carbon::parse($ad->end_date)->addDays($request->extend_days);

        // اضافه کردن یادداشت تمدید به یادداشت‌های قبلی
        $extendNote = 'تمدید در تاریخ '.jdate()->format('Y/m/d').' به مدت '.$request->extend_days.' روز';
        $ad->notes = ($ad->notes ? $ad->notes."\n" : '').$extendNote;

        // اگر تبلیغ غیرفعال بوده، فعالش می‌کنیم
        if ($ad->status != 'active') {
            $ad->status = 'active';
        }

        $ad->save();

        return redirect()->back()->with('success', 'تبلیغ با موفقیت به مدت '.$request->extend_days.' روز تمدید شد.');
    }

    /**
     * حذف تبلیغ
     */
    public function deleteAd(Request $request)
    {
        $ad = Ad::findOrFail($request->ad_id);
        $ad->delete();

        return redirect()->back()->with('success', 'تبلیغ با موفقیت حذف شد.');
    }

    /**
     * نمایش صفحه مدیریت زون‌های نمایش تبلیغات
     */
    public function displayZones()
    {
        $displayZones = DisplayZone::orderBy('display_order', 'asc')->get();

        return view('admin.ads.display-zones', compact('displayZones'));
    }

    /**
     * ایجاد زون نمایش جدید
     */
    public function createZone(Request $request)
    {
        $request->validate([
            'zone_key' => 'required|unique:ad_display_zones,zone_key|regex:/^[a-z0-9_]+$/',
            'title' => 'required|string|max:100',
            'description' => 'nullable|string|max:500',
            'desktop_ad_count' => 'required|integer|min:1|max:20',
            'mobile_ad_count' => 'required|integer|min:1|max:10',
            'display_type' => 'required|in:carousel,grid,banner,list',
            'display_order' => 'required|integer|min:1',
            'is_active' => 'nullable',
        ]);

        $zone = new DisplayZone;
        $zone->zone_key = $request->zone_key;
        $zone->zone_name = $request->title;
        $zone->description = $request->description;
        $zone->position = $request->display_type;
        $zone->max_desktop_cards = $request->desktop_ad_count;
        $zone->max_mobile_cards = $request->mobile_ad_count;
        $zone->card_layout = $request->display_type;
        $zone->display_order = $request->display_order;
        $zone->is_active = $request->has('is_active');
        $zone->created_by = auth()->id();
        $zone->save();

        return redirect()->route('admin.ads.display-zones')
            ->with('success', 'زون نمایش جدید با موفقیت ایجاد شد.');
    }

    /**
     * به‌روزرسانی زون نمایش
     */
    public function updateZone(Request $request)
    {
        $zone = DisplayZone::findOrFail($request->zone_id);

        $request->validate([
            'zone_key' => 'required|regex:/^[a-z0-9_]+$/|unique:ad_display_zones,zone_key,'.$zone->id,
            'title' => 'required|string|max:100',
            'description' => 'nullable|string|max:500',
            'desktop_ad_count' => 'required|integer|min:1|max:20',
            'mobile_ad_count' => 'required|integer|min:1|max:10',
            'display_type' => 'required|in:carousel,grid,banner,list',
            'display_order' => 'required|integer|min:1',
            'is_active' => 'nullable',
        ]);

        $zone->zone_key = $request->zone_key;
        $zone->zone_name = $request->title;
        $zone->description = $request->description;
        $zone->position = $request->display_type;
        $zone->max_desktop_cards = $request->desktop_ad_count;
        $zone->max_mobile_cards = $request->mobile_ad_count;
        $zone->card_layout = $request->display_type;
        $zone->display_order = $request->display_order;
        $zone->is_active = $request->has('is_active');
        $zone->updated_by = auth()->id();
        $zone->save();

        return redirect()->route('admin.ads.display-zones')
            ->with('success', 'زون نمایش با موفقیت به‌روزرسانی شد.');
    }

    /**
     * حذف زون نمایش
     */
    public function deleteZone(Request $request)
    {
        $zone = DisplayZone::findOrFail($request->zone_id);
        $zone->delete();

        return redirect()->route('admin.ads.display-zones')
            ->with('success', 'زون نمایش با موفقیت حذف شد.');
    }

    /**
     * نمایش صفحه مدیریت تخصیص آگهی به زون‌ها
     */
    public function manageZoneAds()
    {
        $zones = DisplayZone::where('is_active', true)
            ->orderBy('display_order')
            ->get();

        $ads = Ad::where('is_active', true)
            ->orderBy('title')
            ->get();

        $locations = Location::where('is_active', true)
            ->orderBy('title')
            ->get();

        // دریافت آگهی‌های تخصیص داده شده به زون‌ها
        $zoneAds = \App\Models\ZoneAd::with(['zone', 'ad', 'location'])
            ->orderBy('zone_id')
            ->orderBy('display_order')
            ->get();

        return view('admin.ads.manage-zone-ads', compact('zones', 'ads', 'locations', 'zoneAds'));
    }

    /**
     * تخصیص آگهی به زون
     */
    public function assignAdToZone(Request $request)
    {
        $request->validate([
            'zone_id' => 'required|exists:ad_display_zones,id',
            'ad_id' => 'required|exists:advertisements,id',
            'location_id' => 'nullable|exists:locations,id',
            'display_order' => 'required|integer|min:0',
            'is_active' => 'nullable',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // بررسی تکراری نبودن آگهی در زون برای همان موقعیت
        $existingAssignment = \App\Models\ZoneAd::where('zone_id', $request->zone_id)
            ->where('ad_id', $request->ad_id)
            ->where('location_id', $request->location_id)
            ->first();

        if ($existingAssignment) {
            return redirect()->back()
                ->with('error', 'این آگهی قبلاً به این زون برای موقعیت مشخص شده تخصیص داده شده است.')
                ->withInput();
        }

        // تخصیص آگهی به زون
        $zoneAd = new \App\Models\ZoneAd;
        $zoneAd->zone_id = $request->zone_id;
        $zoneAd->ad_id = $request->ad_id;
        $zoneAd->location_id = $request->location_id;
        $zoneAd->display_order = $request->display_order;
        $zoneAd->is_active = $request->has('is_active');

        if ($request->start_date) {
            $zoneAd->start_date = $request->start_date;
        }

        if ($request->end_date) {
            $zoneAd->end_date = $request->end_date;
        }

        $zoneAd->created_by = auth()->id();
        $zoneAd->save();

        return redirect()->route('admin.ads.manage-zone-ads')
            ->with('success', 'آگهی با موفقیت به زون مورد نظر تخصیص داده شد.');
    }

    /**
     * به‌روزرسانی تخصیص آگهی به زون
     */
    public function updateZoneAdAssignment(Request $request)
    {
        $request->validate([
            'zone_ad_id' => 'required|exists:zone_ads,id',
            'display_order' => 'required|integer|min:0',
            'is_active' => 'nullable',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $zoneAd = \App\Models\ZoneAd::findOrFail($request->zone_ad_id);
        $zoneAd->display_order = $request->display_order;
        $zoneAd->is_active = $request->has('is_active');

        if ($request->filled('start_date')) {
            $zoneAd->start_date = $request->start_date;
        } else {
            $zoneAd->start_date = null;
        }

        if ($request->filled('end_date')) {
            $zoneAd->end_date = $request->end_date;
        } else {
            $zoneAd->end_date = null;
        }

        $zoneAd->updated_by = auth()->id();
        $zoneAd->save();

        return redirect()->route('admin.ads.manage-zone-ads')
            ->with('success', 'تنظیمات آگهی در زون مورد نظر با موفقیت به‌روزرسانی شد.');
    }

    /**
     * حذف تخصیص آگهی از زون
     */
    public function deleteZoneAdAssignment(Request $request)
    {
        $zoneAd = \App\Models\ZoneAd::findOrFail($request->zone_ad_id);
        $zoneAd->delete();

        return redirect()->route('admin.ads.manage-zone-ads')
            ->with('success', 'تخصیص آگهی به زون با موفقیت حذف شد.');
    }
}
