<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Language;
use App\Services\TranslationService;
use Illuminate\Http\Request;

/**
 * کنترلر مدیریت ترجمه‌ها
 * این کنترلر عملیات CRUD و مدیریت ترجمه‌های دیتابیسی را انجام می‌دهد.
 *
 * @author AI
 */
class TranslationController extends Controller
{
    protected $service;

    /**
     * سازنده و تزریق سرویس ترجمه
     */
    public function __construct(TranslationService $service)
    {
        $this->service = $service;
    }

    /**
     * نمایش لیست ترجمه‌ها با فیلتر و جستجو
     */
    public function index(Request $request)
    {
        // دریافت فیلترها
        $filters = $request->only(['search', 'language', 'group', 'namespace']);
        $translations = $this->service->getPaginatedTranslations($filters);
        $languages = Language::where('is_active', 1)->get();
        $groups = $this->service->getGroups();
        $namespaces = $this->service->getNamespaces();

        // اگر درخواست ajax باشد، فقط داده‌ها را برگردان
        if ($request->ajax()) {
            return response()->json([
                'translations' => $translations,
                'pagination' => [
                    'current_page' => $translations->currentPage(),
                    'last_page' => $translations->lastPage(),
                ],
            ]);
        }

        return view('admin.translations.index', compact('translations', 'languages', 'groups', 'namespaces'));
    }

    /**
     * نمایش فرم ایجاد ترجمه جدید
     */
    public function create()
    {
        $languages = Language::where('is_active', 1)->get();

        return view('admin.translations.create', compact('languages'));
    }

    /**
     * ذخیره ترجمه جدید
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'key' => 'required|string|max:255',
            'value' => 'required|string',
            'language_code' => 'required|exists:languages,code',
            'group' => 'nullable|string|max:50',
            'namespace' => 'nullable|string|max:50',
        ]);
        $this->service->createTranslation($validated);

        return redirect()->route('admin.translations.index')->with('success', 'ترجمه با موفقیت ثبت شد.');
    }

    /**
     * نمایش فرم ویرایش ترجمه
     */
    public function edit($id)
    {
        $translation = $this->service->findTranslation($id);

        return view('admin.translations.edit', compact('translation'));
    }

    /**
     * بروزرسانی ترجمه
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'value' => 'required|string',
        ]);
        $this->service->updateTranslation($id, $validated);

        return redirect()->route('admin.translations.index')->with('success', 'ترجمه با موفقیت بروزرسانی شد.');
    }

    /**
     * حذف ترجمه
     */
    public function destroy($id)
    {
        $this->service->deleteTranslation($id);

        return redirect()->route('admin.translations.index')->with('success', 'ترجمه حذف شد.');
    }

    /**
     * ایمپورت ترجمه‌ها از فایل‌های زبان
     */
    public function import()
    {
        $this->service->importFromFiles();

        return redirect()->route('admin.translations.index')->with('success', 'ایمپورت ترجمه‌ها با موفقیت انجام شد.');
    }

    /**
     * اکسپورت ترجمه‌ها به فایل
     */
    public function export()
    {
        return $this->service->exportToFile();
    }

    /**
     * اسکن فایل‌های زبان برای کلیدهای جدید
     */
    public function scan()
    {
        $this->service->scanForNewKeys();

        return redirect()->route('admin.translations.index')->with('success', 'اسکن کلیدهای جدید با موفقیت انجام شد.');
    }
}
