<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckRequestSize
{
    protected $maxRequestSize = 10485760; // 10MB in bytes

    protected $maxFileSize = 5242880; // 5MB in bytes

    protected $maxFiles = 5;

    public function handle(Request $request, Closure $next): Response
    {
        // Check total request size
        if ($request->header('Content-Length') > $this->maxRequestSize) {
            abort(413, 'حجم درخواست بیش از حد مجاز است.');
        }

        // Check file uploads
        if ($request->hasFile('files')) {
            $files = $request->file('files');

            if (count($files) > $this->maxFiles) {
                abort(413, 'تعداد فایل‌های ارسالی بیش از حد مجاز است.');
            }

            foreach ($files as $file) {
                if ($file->getSize() > $this->maxFileSize) {
                    abort(413, 'حجم یکی از فایل‌های ارسالی بیش از حد مجاز است.');
                }

                if (! $this->isAllowedFileType($file)) {
                    abort(415, 'نوع فایل ارسالی مجاز نیست.');
                }
            }
        }

        return $next($request);
    }

    protected function isAllowedFileType($file): bool
    {
        $allowedMimes = [
            'image/jpeg',
            'image/png',
            'image/gif',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/plain',
            'text/csv',
        ];

        return in_array($file->getMimeType(), $allowedMimes);
    }
}
