<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;

class ConcurrentRequestsLimit
{
    protected $maxConcurrentRequests = 10; // حداکثر تعداد درخواست همزمان

    protected $lockTimeout = 30; // زمان قفل به ثانیه

    public function handle(Request $request, Closure $next): Response
    {
        $ip = $request->ip();
        $cacheKey = "concurrent_requests:{$ip}";

        // دریافت تعداد درخواست‌های فعلی
        $currentRequests = Cache::get($cacheKey, 0);

        if ($currentRequests >= $this->maxConcurrentRequests) {
            abort(429, 'تعداد درخواست‌های همزمان بیش از حد مجاز است. لطفاً کمی صبر کنید.');
        }

        // افزایش شمارنده
        Cache::put($cacheKey, $currentRequests + 1, $this->lockTimeout);

        try {
            $response = $next($request);

            return $response;
        } finally {
            // کاهش شمارنده پس از اتمام درخواست
            $currentRequests = Cache::get($cacheKey, 0);
            if ($currentRequests > 0) {
                Cache::put($cacheKey, $currentRequests - 1, $this->lockTimeout);
            }
        }
    }
}
