<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class EnhancedCsrfProtection
{
    /**
     * زمان اعتبار توکن CSRF (به ثانیه)
     */
    protected $tokenLifetime = 7200; // 2 ساعت

    /**
     * تعداد دفعات مجاز تلاش برای توکن نامعتبر
     */
    protected $maxInvalidAttempts = 5;

    /**
     * زمان قفل شدن پس از تلاش‌های ناموفق (به ثانیه)
     */
    protected $lockoutTime = 1800; // 30 دقیقه

    public function handle(Request $request, Closure $next): Response
    {
        // بررسی متد درخواست
        if ($this->isReadingMethod($request)) {
            return $next($request);
        }

        // بررسی تعداد تلاش‌های ناموفق
        if ($this->isLockedOut($request)) {
            $this->logLockoutAttempt($request);

            return response()->json([
                'message' => 'به دلیل تلاش‌های مکرر ناموفق، دسترسی شما موقتاً مسدود شده است.',
                'retry_after' => $this->getLockoutTimeLeft($request),
            ], 429);
        }

        // بررسی توکن CSRF
        if (! $this->tokensMatch($request)) {
            $this->incrementInvalidAttempts($request);
            $this->logInvalidToken($request);

            return response()->json([
                'message' => 'توکن امنیتی نامعتبر است.',
                'error_code' => 'invalid_csrf_token',
            ], 419);
        }

        // بررسی زمان اعتبار توکن
        if ($this->isTokenExpired($request)) {
            $this->logExpiredToken($request);

            return response()->json([
                'message' => 'توکن امنیتی منقضی شده است. لطفاً صفحه را رفرش کنید.',
                'error_code' => 'expired_csrf_token',
            ], 419);
        }

        // تنظیم هدرهای امنیتی
        $response = $next($request);
        $this->setSecurityHeaders($response);

        return $response;
    }

    protected function isReadingMethod(Request $request): bool
    {
        return in_array($request->method(), ['HEAD', 'GET', 'OPTIONS']);
    }

    protected function tokensMatch(Request $request): bool
    {
        $token = $request->input('_token') ?: $request->header('X-CSRF-TOKEN');

        // Check for XSRF token in cookies
        if (! $token && $header = $request->header('X-XSRF-TOKEN')) {
            try {
                $token = decrypt($header, false);
            } catch (\Exception $e) {
                $this->logInvalidToken($request, 'Invalid XSRF token decryption');
                return false;
            }
        }

        // Additional validation for token format
        if (!$token || !is_string($token) || strlen($token) < 40) {
            $this->logInvalidToken($request, 'Invalid token format');
            return false;
        }

        $sessionToken = $request->session()->token();
        if (!$sessionToken || !is_string($sessionToken)) {
            $this->logInvalidToken($request, 'Invalid session token');
            return false;
        }

        return hash_equals($sessionToken, $token);
    }

    protected function isTokenExpired(Request $request): bool
    {
        $tokenTime = $request->session()->get('csrf_token_time');

        if (! $tokenTime) {
            return true;
        }

        return time() - $tokenTime > $this->tokenLifetime;
    }

    protected function isLockedOut(Request $request): bool
    {
        $attempts = $request->session()->get('csrf_invalid_attempts', 0);
        $lockoutTime = $request->session()->get('csrf_lockout_time');

        if ($attempts >= $this->maxInvalidAttempts && $lockoutTime) {
            return time() - $lockoutTime < $this->lockoutTime;
        }

        return false;
    }

    protected function incrementInvalidAttempts(Request $request): void
    {
        $attempts = $request->session()->get('csrf_invalid_attempts', 0) + 1;
        $request->session()->put('csrf_invalid_attempts', $attempts);

        if ($attempts >= $this->maxInvalidAttempts) {
            $request->session()->put('csrf_lockout_time', time());
        }
    }

    protected function getLockoutTimeLeft(Request $request): int
    {
        $lockoutTime = $request->session()->get('csrf_lockout_time');

        return max(0, $this->lockoutTime - (time() - $lockoutTime));
    }

    protected function setSecurityHeaders(Response $response): void
    {
        $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
        $response->headers->set('X-XSS-Protection', '1; mode=block');
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
    }

    protected function logInvalidToken(Request $request, string $reason = 'Invalid token'): void
    {
        Log::warning('تلاش با توکن CSRF نامعتبر', [
            'ip' => $request->ip(),
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
            'reason' => $reason,
            'attempts' => $request->session()->get('csrf_invalid_attempts'),
            'referer' => $request->header('Referer'),
            'timestamp' => now()->toISOString(),
        ]);
    }

    protected function logExpiredToken(Request $request): void
    {
        Log::warning('تلاش با توکن CSRF منقضی شده', [
            'ip' => $request->ip(),
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
        ]);
    }

    protected function logLockoutAttempt(Request $request): void
    {
        Log::warning('تلاش دسترسی در زمان قفل بودن CSRF', [
            'ip' => $request->ip(),
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
            'lockout_time_left' => $this->getLockoutTimeLeft($request),
        ]);
    }
}
