<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class PathTraversalProtection
{
    /**
     * Path traversal patterns to detect
     */
    protected $pathTraversalPatterns = [
        // Basic patterns
        '/\.\.\//',
        '/\.\.\\\\/',
        '/\.\.\%2f/',
        '/\.\.\%5c/',
        '/\.\.\%252f/',
        '/\.\.\%255c/',
        
        // URL encoded patterns
        '/\%2e\%2e\%2f/',
        '/\%2e\%2e\%5c/',
        '/\%2e\%2e\//',
        '/\%2e\%2e\\\\/',
        
        // Double URL encoded
        '/\%252e\%252e\%252f/',
        '/\%252e\%252e\%255c/',
        
        // Unicode encoded
        '/\%c0\%ae\%c0\%ae\%c0\%af/',
        '/\%c1\%9c/',
        '/\%c0\%af/',
        
        // Alternative representations
        '/\.\.\//',
        '/\.\.\\\\/',
        '/\.\.\x2f/',
        '/\.\.\x5c/',
        
        // File system specific
        '/\/etc\/passwd/',
        '/\/etc\/shadow/',
        '/\/etc\/hosts/',
        '/\/proc\//',
        '/\/sys\//',
        '/\/dev\//',
        '/\/var\/log\//',
        '/\/tmp\//',
        '/\/boot\//',
        '/\/root\//',
        '/\/home\//',
        '/\/usr\//',
        '/\/opt\//',
        '/\/mnt\//',
        '/\/media\//',
        
        // Windows specific
        '/c:\\\\/i',
        '/d:\\\\/i',
        '/\\\\windows\\\\/i',
        '/\\\\system32\\\\/i',
        '/\\\\program files\\\\/i',
        '/\\\\users\\\\/i',
        '/\\\\documents and settings\\\\/i',
        '/boot\.ini/i',
        '/win\.ini/i',
        '/system\.ini/i',
        '/autoexec\.bat/i',
        '/config\.sys/i',
        
        // Application specific
        '/\.htaccess/',
        '/\.htpasswd/',
        '/\.env/',
        '/\.git\//',
        '/\.svn\//',
        '/\.hg\//',
        '/\.bzr\//',
        '/web\.config/',
        '/composer\.json/',
        '/package\.json/',
        '/\.ssh\//',
        '/\.aws\//',
        '/\.docker\//',
        
        // Database files
        '/\.db$/',
        '/\.sqlite$/',
        '/\.mdb$/',
        '/\.accdb$/',
        
        // Backup files
        '/\.bak$/',
        '/\.backup$/',
        '/\.old$/',
        '/\.orig$/',
        '/\.tmp$/',
        '/\.temp$/',
        '/~$/',
        '/\.swp$/',
        '/\.swo$/',
        
        // Log files
        '/\.log$/',
        '/\.out$/',
        '/\.err$/',
        
        // Configuration files
        '/\.conf$/',
        '/\.cfg$/',
        '/\.ini$/',
        '/\.xml$/',
        '/\.yml$/',
        '/\.yaml$/',
    ];

    /**
     * Dangerous file extensions
     */
    protected $dangerousExtensions = [
        'php', 'php3', 'php4', 'php5', 'phtml', 'phps',
        'asp', 'aspx', 'jsp', 'jspx',
        'pl', 'py', 'rb', 'sh', 'bash',
        'exe', 'bat', 'cmd', 'com', 'scr',
        'vbs', 'vbe', 'js', 'jar',
        'htaccess', 'htpasswd'
    ];

    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Check for path traversal in URL
        $this->checkUrlPath($request);

        // Check for path traversal in parameters
        $this->checkParameters($request);

        // Check for path traversal in headers
        $this->checkHeaders($request);

        // Check file uploads
        $this->checkFileUploads($request);

        return $next($request);
    }

    /**
     * Check URL path for traversal patterns
     */
    protected function checkUrlPath(Request $request): void
    {
        $path = $request->getPathInfo();
        $decodedPath = urldecode($path);
        
        if ($this->containsPathTraversal($decodedPath)) {
            $this->logPathTraversalAttempt($request, 'URL_PATH', 'path', $path);
            abort(403, 'Path traversal attempt detected');
        }
    }

    /**
     * Check request parameters for traversal patterns
     */
    protected function checkParameters(Request $request): void
    {
        // Check GET parameters
        $this->checkData($request->query->all(), $request, 'GET');

        // Check POST data
        $this->checkData($request->request->all(), $request, 'POST');

        // Check JSON data
        if ($request->isJson()) {
            $this->checkData($request->json()->all(), $request, 'JSON');
        }
    }

    /**
     * Check data array for path traversal patterns
     */
    protected function checkData(array $data, Request $request, string $type): void
    {
        array_walk_recursive($data, function ($value, $key) use ($request, $type) {
            if (is_string($value) && $this->containsPathTraversal($value)) {
                $this->logPathTraversalAttempt($request, $type, $key, $value);
                abort(403, 'Path traversal attempt detected');
            }
        });
    }

    /**
     * Check headers for path traversal patterns
     */
    protected function checkHeaders(Request $request): void
    {
        $suspiciousHeaders = [
            'X-Original-URL',
            'X-Rewrite-URL',
            'X-Forwarded-Host',
            'X-Forwarded-Server',
            'X-HTTP-Host-Override',
            'Forwarded'
        ];

        foreach ($suspiciousHeaders as $header) {
            if ($request->hasHeader($header)) {
                $value = $request->header($header);
                if ($this->containsPathTraversal($value)) {
                    $this->logPathTraversalAttempt($request, 'HEADER', $header, $value);
                    abort(403, 'Path traversal attempt detected in headers');
                }
            }
        }
    }

    /**
     * Check file uploads for dangerous patterns
     */
    protected function checkFileUploads(Request $request): void
    {
        if ($request->hasFile('*')) {
            foreach ($request->allFiles() as $key => $files) {
                $files = is_array($files) ? $files : [$files];
                
                foreach ($files as $file) {
                    if ($file && $file->isValid()) {
                        $originalName = $file->getClientOriginalName();
                        $extension = strtolower($file->getClientOriginalExtension());
                        
                        // Check filename for traversal patterns
                        if ($this->containsPathTraversal($originalName)) {
                            $this->logPathTraversalAttempt($request, 'FILE_UPLOAD', 'filename', $originalName);
                            abort(403, 'Dangerous filename detected');
                        }
                        
                        // Check for dangerous extensions
                        if (in_array($extension, $this->dangerousExtensions)) {
                            $this->logPathTraversalAttempt($request, 'FILE_UPLOAD', 'extension', $extension);
                            abort(403, 'Dangerous file extension detected');
                        }
                        
                        // Check for null bytes in filename
                        if (strpos($originalName, "\0") !== false) {
                            $this->logPathTraversalAttempt($request, 'FILE_UPLOAD', 'null_byte', $originalName);
                            abort(403, 'Null byte detected in filename');
                        }
                    }
                }
            }
        }
    }

    /**
     * Check if string contains path traversal patterns
     */
    protected function containsPathTraversal(string $value): bool
    {
        // Skip check for very short strings
        if (strlen($value) < 2) {
            return false;
        }

        // URL decode multiple times to catch double encoding
        $decodedValue = $value;
        for ($i = 0; $i < 3; $i++) {
            $decodedValue = urldecode($decodedValue);
        }
        
        // Convert to lowercase for case-insensitive matching
        $lowerValue = strtolower($decodedValue);
        
        foreach ($this->pathTraversalPatterns as $pattern) {
            if (preg_match($pattern, $lowerValue)) {
                return true;
            }
        }

        // Check for consecutive dots and slashes
        if (preg_match('/\.{2,}/', $decodedValue) && preg_match('/[\/\\\\]/', $decodedValue)) {
            return true;
        }

        // Check for null bytes
        if (strpos($decodedValue, "\0") !== false) {
            return true;
        }

        return false;
    }

    /**
     * Log path traversal attempt
     */
    protected function logPathTraversalAttempt(Request $request, string $type, string $field, string $value): void
    {
        Log::warning('Path traversal attempt detected', [
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'url' => $request->fullUrl(),
            'method' => $request->method(),
            'type' => $type,
            'field' => $field,
            'value' => substr($value, 0, 200), // Limit logged value length
            'referer' => $request->header('Referer'),
            'timestamp' => now()->toISOString(),
        ]);
    }
}
