<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class Ad extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'advertisements';

    protected $fillable = [
        'title',
        'description',
        'ad_type',
        'target_url',
        'image_path',
        'mobile_image_path',
        'video_path',
        'is_active',
        'view_count',
        'click_count',
        'start_date',
        'end_date',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'view_count' => 'integer',
        'click_count' => 'integer',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    /**
     * رابطه با زون‌های نمایش
     */
    public function zones()
    {
        return $this->belongsToMany(DisplayZone::class, 'zone_ads', 'ad_id', 'zone_id')
            ->withPivot('location_id', 'display_order', 'is_active', 'start_date', 'end_date')
            ->withTimestamps();
    }

    /**
     * تنظیمات آگهی در زون‌ها
     */
    public function zoneSettings()
    {
        return $this->hasMany(ZoneAd::class, 'ad_id');
    }

    /**
     * رابطه با کاربر ایجاد کننده
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * رابطه با کاربر ویرایش کننده
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * فیلتر کردن آگهی‌های فعال
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
                $q->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($q) {
                $q->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });
    }

    /**
     * آدرس کامل تصویر آگهی
     */
    public function getImageUrlAttribute()
    {
        if (! $this->image_path) {
            return null;
        }

        return Storage::url($this->image_path);
    }

    /**
     * آدرس کامل تصویر موبایل آگهی
     */
    public function getMobileImageUrlAttribute()
    {
        if (! $this->mobile_image_path) {
            return $this->getImageUrlAttribute();
        }

        return Storage::url($this->mobile_image_path);
    }

    /**
     * آدرس کامل ویدیوی آگهی
     */
    public function getVideoUrlAttribute()
    {
        if (! $this->video_path) {
            return null;
        }

        return Storage::url($this->video_path);
    }

    /**
     * افزایش تعداد بازدید آگهی
     */
    public function incrementViewCount()
    {
        $this->increment('view_count');
    }

    /**
     * افزایش تعداد کلیک آگهی
     */
    public function incrementClickCount()
    {
        $this->increment('click_count');
    }
}
