<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Rating extends Model
{
    use HasFactory;

    /**
     * جدول مورد استفاده توسط مدل
     *
     * @var string
     */
    protected $table = 'ratings';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = true;

    /**
     * The name of the "created at" column.
     *
     * @var string
     */
    const CREATED_AT = 'created_at';

    /**
     * The name of the "updated at" column.
     *
     * @var string
     */
    const UPDATED_AT = 'updated_at';

    /**
     * The primary key associated with the table.
     *
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * ویژگی‌هایی که اجازه تخصیص انبوه را دارند
     *
     * @var array
     */
    protected $fillable = [
        'title_id',
        'category',
        'hotel_name',
        'name',
        'email',
        'rating',
        'service_rating',
        'cleanliness_rating',
        'location_rating',
        'price_rating',
        'quality_rating',
        'positive_comment',
        'negative_comment',
        'comment',
        'trip_type',
        'travel_method',
        'stay_date',
        'visit_date',
        'recommendation',
        'status',
        'timestamp',
        'created_at',
        'updated_at'
    ];

    /**
     * ویژگی‌هایی که باید به نوع داده تبدیل شوند
     *
     * @var array
     */
    protected $casts = [
        'stay_date' => 'date',
        'timestamp' => 'datetime',
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'timestamp',
        'stay_date',
    ];

    /**
     * رابطه با مکان مرتبط با نظر
     */
    public function location()
    {
        return $this->belongsTo(Location::class, 'title_id', 'title_id');
    }

    /**
     * رابطه با دسته‌بندی مرتبط با نظر
     */
    public function categoryRelation()
    {
        return $this->belongsTo(Category::class, 'category', 'category_key');
    }

    /**
     * دریافت نام دسته‌بندی بر اساس زبان فعال
     */
    public function getLocalizedCategoryName()
    {
        if ($this->categoryRelation) {
            return $this->categoryRelation->getTranslatedName();
        }

        // Fallback to the original category value if no relation found
        return $this->category;
    }

    /**
     * نظرات تایید شده
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * نظرات در انتظار تایید
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * دریافت میانگین امتیازات برای یک مکان خاص
     *
     * @param  int  $locationId
     * @return float
     */
    public static function getAverageRating($locationId)
    {
        $location = Location::find($locationId);
        if (! $location) {
            return 0;
        }

        return static::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->avg('rating') ?? 0;
    }

    /**
     * دریافت تعداد امتیازات برای یک مکان خاص
     *
     * @param  int  $locationId
     * @return int
     */
    public static function getRatingCount($locationId)
    {
        $location = Location::find($locationId);
        if (! $location) {
            return 0;
        }

        return static::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->count();
    }

    /**
     * دریافت توزیع امتیازات برای یک مکان خاص
     *
     * @param  int  $locationId
     * @return array
     */
    public static function getRatingDistribution($locationId)
    {
        $distribution = [5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0];

        $location = Location::find($locationId);
        if (! $location) {
            return $distribution;
        }

        $ratings = static::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->get(['rating']);

        foreach ($ratings as $rating) {
            if (isset($distribution[$rating->rating])) {
                $distribution[$rating->rating]++;
            }
        }

        return $distribution;
    }

    /**
     * Get positive comment formatted as HTML list
     */
    public function getPositiveCommentHtmlAttribute()
    {
        return $this->formatCommentAsHtml($this->positive_comment, 'positive');
    }

    /**
     * Get negative comment formatted as HTML list
     */
    public function getNegativeCommentHtmlAttribute()
    {
        return $this->formatCommentAsHtml($this->negative_comment, 'negative');
    }

    /**
     * Format comment text as HTML list with appropriate icons
     */
    private function formatCommentAsHtml($comment, $type = 'positive')
    {
        if (empty($comment)) {
            return null;
        }

        // Split comment by lines and remove empty lines
        $lines = array_filter(array_map('trim', explode("\n", $comment)));

        if (empty($lines)) {
            return null;
        }

        // Choose icon based on type
        $icon = $type === 'positive' ?
            '<i class="fas fa-plus-circle text-success me-2"></i>' :
            '<i class="fas fa-minus-circle text-danger me-2"></i>';

        // Convert each line to a list item
        $listItems = array_map(function($line) use ($icon) {
            return '<li class="comment-list-item">' . $icon . htmlspecialchars($line) . '</li>';
        }, $lines);

        // Return as HTML list
        return '<ul class="comment-list">' . implode('', $listItems) . '</ul>';
    }
}
