<?php

namespace App\Services;

use App\Models\SiteSetting;
use Illuminate\Support\Facades\Cache;

class SiteSettingService
{
    /**
     * دریافت تنظیمات سایت برای زبان مشخص
     *
     * @param string $languageCode
     * @return array
     */
    public function getSettings($languageCode = null)
    {
        if (!$languageCode) {
            $languageCode = app()->getLocale();
        }

        return Cache::remember('site_settings_' . $languageCode, 3600, function () use ($languageCode) {
            $settings = SiteSetting::where('language_code', $languageCode)
                ->where('is_active', true)
                ->get();

            $result = [];
            foreach ($settings as $setting) {
                $result[$setting->setting_key] = $setting->setting_value;
            }

            return $result;
        });
    }

    /**
     * دریافت یک تنظیم خاص
     *
     * @param string $key
     * @param string $languageCode
     * @param mixed $default
     * @return mixed
     */
    public function getSetting($key, $languageCode = null, $default = null)
    {
        $settings = $this->getSettings($languageCode);
        return $settings[$key] ?? $default;
    }

    /**
     * ذخیره تنظیمات جدید
     *
     * @param array $data
     * @param string $languageCode
     * @return void
     */
    public function saveSettings($data, $languageCode)
    {
        foreach ($data as $key => $value) {
            // اگر مقدار null است، تنظیم را حذف می‌کنیم
            if ($value === null) {
                SiteSetting::where('setting_key', $key)
                    ->where('language_code', $languageCode)
                    ->delete();
                continue;
            }

            // ذخیره تنظیم جدید
            SiteSetting::setSetting($key, $value, $languageCode);
        }

        $this->clearCache($languageCode);
    }

    /**
     * پاک کردن کش تنظیمات
     *
     * @param string $languageCode
     * @return void
     */
    public function clearCache($languageCode = null)
    {
        if ($languageCode) {
            Cache::forget('site_settings_' . $languageCode);
        } else {
            $languages = \App\Models\Language::where('is_active', true)->get();
            foreach ($languages as $language) {
                Cache::forget('site_settings_' . $language->code);
            }
        }
    }

    /**
     * تغییر وضعیت حالت تعمیر و نگهداری
     *
     * @param string $languageCode
     * @return bool
     */
    public function toggleMaintenanceMode($languageCode)
    {
        $maintenanceMode = $this->getSetting('maintenance_mode', $languageCode, false);
        $newValue = !$maintenanceMode;
        
        SiteSetting::setSetting('maintenance_mode', $newValue, $languageCode);
        $this->clearCache($languageCode);
        
        return $newValue;
    }

    /**
     * بررسی وضعیت حالت تعمیر و نگهداری
     *
     * @param string $languageCode
     * @return bool
     */
    public function isMaintenanceMode($languageCode = null)
    {
        return (bool) $this->getSetting('maintenance_mode', $languageCode, false);
    }

    /**
     * دریافت پیام حالت تعمیر و نگهداری
     *
     * @param string $languageCode
     * @return string
     */
    public function getMaintenanceMessage($languageCode = null)
    {
        return $this->getSetting('maintenance_message', $languageCode);
    }

    /**
     * دریافت تنظیمات SMTP
     *
     * @param string $languageCode
     * @return array
     */
    public function getSmtpSettings(string $languageCode): array
    {
        return $this->getSetting('smtp_settings', $languageCode, []);
    }

    /**
     * دریافت تنظیمات پرداخت
     *
     * @param string $languageCode
     * @return array
     */
    public function getPaymentSettings(string $languageCode): array
    {
        return $this->getSetting('payment_settings', $languageCode, []);
    }

    /**
     * دریافت تنظیمات اعلان‌ها
     *
     * @param string $languageCode
     * @return array
     */
    public function getNotificationSettings(string $languageCode): array
    {
        return $this->getSetting('notification_settings', $languageCode, []);
    }

    /**
     * دریافت تنظیمات SEO
     *
     * @param string $languageCode
     * @return array
     */
    public function getSeoSettings(string $languageCode): array
    {
        return [
            'meta_title' => $this->getSetting('meta_title', $languageCode),
            'meta_description' => $this->getSetting('meta_description', $languageCode),
            'meta_keywords' => $this->getSetting('meta_keywords', $languageCode),
        ];
    }

    /**
     * دریافت تنظیمات نمایش
     *
     * @param string $languageCode
     * @return array
     */
    public function getDisplaySettings(string $languageCode): array
    {
        return [
            'theme_color' => $this->getSetting('theme_color', $languageCode),
            'custom_css' => $this->getSetting('custom_css', $languageCode),
            'custom_js' => $this->getSetting('custom_js', $languageCode),
            'logo' => $this->getSetting('logo', $languageCode),
            'logo_text' => $this->getSetting('logo_text', $languageCode),
            'site_icon' => $this->getSetting('site_icon', $languageCode),
            'favicon' => $this->getSetting('favicon', $languageCode),
        ];
    }

    /**
     * دریافت تنظیمات شبکه‌های اجتماعی
     *
     * @param string $languageCode
     * @return array
     */
    public function getSocialMediaSettings(string $languageCode): array
    {
        return $this->getSetting('social_media', $languageCode, []);
    }

    /**
     * دریافت اطلاعات تماس
     *
     * @param string $languageCode
     * @return array
     */
    public function getContactInfo(string $languageCode): array
    {
        return [
            'email' => $this->getSetting('contact_email', $languageCode),
            'phone' => $this->getSetting('contact_phone', $languageCode),
            'address' => $this->getSetting('contact_address', $languageCode),
        ];
    }
} 