/**
 * Rating System JavaScript
 * Handles interactive star ratings and form functionality
 */

document.addEventListener('DOMContentLoaded', function() {
    // Star rating functionality
    initializeStarRating();
    
    // Form validation and submission
    initializeRatingForm();
});

/**
 * Initialize star rating functionality
 */
function initializeStarRating() {
    // Interactive stars for new ratings
    const stars = document.querySelectorAll('.rating-label');
    const inputs = document.querySelectorAll('input[name="rating"]');
    
    if (stars.length === 0) return;
    
    // Check if any stars are already selected (from old input)
    const selectedInput = Array.from(inputs).find(input => input.checked);
    if (selectedInput) {
        const selectedValue = selectedInput.value;
        stars.forEach(star => {
            const value = star.getAttribute('for').replace('star', '');
            const icon = star.querySelector('.star-icon');
            
            if (value <= selectedValue) {
                icon.classList.remove('far');
                icon.classList.add('fas');
            }
        });
    }
    
    stars.forEach(star => {
        // Mouse hover effects
        star.addEventListener('mouseenter', function() {
            const starValue = this.getAttribute('for').replace('star', '');
            
            stars.forEach(s => {
                const value = s.getAttribute('for').replace('star', '');
                const icon = s.querySelector('.star-icon');
                
                if (value <= starValue) {
                    icon.classList.remove('far');
                    icon.classList.add('fas');
                } else {
                    icon.classList.remove('fas');
                    icon.classList.add('far');
                }
            });
        });
        
        // Click events
        star.addEventListener('click', function() {
            const starValue = this.getAttribute('for').replace('star', '');
            document.getElementById('star' + starValue).checked = true;
            
            stars.forEach(s => {
                const value = s.getAttribute('for').replace('star', '');
                const icon = s.querySelector('.star-icon');
                
                if (value <= starValue) {
                    icon.classList.remove('far');
                    icon.classList.add('fas');
                } else {
                    icon.classList.remove('fas');
                    icon.classList.add('far');
                }
            });
        });
    });
    
    // Restore selected state when mouse leaves the container
    const container = document.querySelector('.rating-stars-input');
    if (container) {
        container.addEventListener('mouseleave', function() {
            const selectedInput = Array.from(inputs).find(input => input.checked);
            const selectedValue = selectedInput ? selectedInput.value : 0;
            
            stars.forEach(s => {
                const value = s.getAttribute('for').replace('star', '');
                const icon = s.querySelector('.star-icon');
                
                if (value <= selectedValue) {
                    icon.classList.remove('far');
                    icon.classList.add('fas');
                } else {
                    icon.classList.remove('fas');
                    icon.classList.add('far');
                }
            });
        });
    }
}

/**
 * Initialize rating form validation and submission
 */
function initializeRatingForm() {
    const form = document.getElementById('rating-form');
    if (!form) return;
    
    form.addEventListener('submit', function(e) {
        // Basic validation
        const rating = document.querySelector('input[name="rating"]:checked');
        const name = document.getElementById('name');
        
        if (!rating) {
            e.preventDefault();
            alert('لطفاً یک امتیاز انتخاب کنید');
            return false;
        }
        
        if (!name.value.trim()) {
            e.preventDefault();
            alert('لطفاً نام خود را وارد کنید');
            name.focus();
            return false;
        }
        
        // Add any additional validation here
        
        // If all validation passes, the form will submit naturally
    });
}

// Rating widget display
const ratingDisplays = document.querySelectorAll('.rating-display');
ratingDisplays.forEach(display => {
    const rating = parseFloat(display.dataset.rating);
    const stars = display.querySelectorAll('.star-icon');
    
    stars.forEach((star, index) => {
        if (index < Math.floor(rating)) {
            star.classList.remove('far');
            star.classList.add('fas');
        } else if (index < Math.ceil(rating) && rating % 1 !== 0) {
            star.classList.remove('far', 'fas');
            star.classList.add('fas', 'fa-star-half-alt');
        }
    });
}); 