<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Models\LocationContent;
use App\Models\LocationContentType;
use Illuminate\Http\Request;

class LocationContentController extends Controller
{
    /**
     * Display a listing of the location contents.
     */
    public function index(?Location $location = null)
    {
        $query = LocationContent::with('contentType');

        if ($location) {
            $query->where('location_id', $location->id);
        }

        $contents = $query->orderBy('sort_order')->paginate(15);

        return view('admin.location_contents.index', compact('contents', 'location'));
    }

    /**
     * Show the form for creating a new content.
     */
    public function create(?Location $location = null)
    {
        $contentTypes = LocationContentType::where('is_active', 1)->orderBy('display_order')->get();
        $locations = Location::where('is_active', 1)->orderBy('title')->get();

        return view('admin.location_contents.create', compact('contentTypes', 'locations', 'location'));
    }

    /**
     * Store a newly created content in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'location_id' => 'required|exists:locations,id',
            'content_type_id' => 'required|exists:location_content_types,id',
            'title' => 'nullable|string|max:255',
            'content' => 'nullable|string',
            'sort_order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
        ]);

        LocationContent::create($validated);

        return redirect()->route('locations.contents.index', $request->location_id)
            ->with('success', 'محتوا با موفقیت ایجاد شد');
    }

    /**
     * Show the form for editing the specified content.
     */
    public function edit(LocationContent $locationContent)
    {
        $contentTypes = LocationContentType::where('is_active', 1)->orderBy('display_order')->get();
        $locations = Location::where('is_active', 1)->orderBy('title')->get();

        return view('admin.location_contents.edit', compact('locationContent', 'contentTypes', 'locations'));
    }

    /**
     * Update the specified content in storage.
     */
    public function update(Request $request, LocationContent $locationContent)
    {
        $validated = $request->validate([
            'location_id' => 'required|exists:locations,id',
            'content_type_id' => 'required|exists:location_content_types,id',
            'title' => 'nullable|string|max:255',
            'content' => 'nullable|string',
            'sort_order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
        ]);

        $locationContent->update($validated);

        return redirect()->route('locations.contents.index', $request->location_id)
            ->with('success', 'محتوا با موفقیت بروزرسانی شد');
    }

    /**
     * Remove the specified content from storage.
     */
    public function destroy(LocationContent $locationContent)
    {
        $locationId = $locationContent->location_id;
        $locationContent->delete();

        return redirect()->route('locations.contents.index', $locationId)
            ->with('success', 'محتوا با موفقیت حذف شد');
    }
}
