<?php

namespace App\Services;

use App\Models\AdDisplayZone;
use App\Models\AdPackage;
use App\Models\DisplayZone;
use App\Models\Location;
use App\Models\LocationAdPackage;
use App\Models\ZoneAd;

class AdService
{
    /**
     * دریافت تبلیغات برای ناحیه خاص
     */
    public function getAdsForZone($zoneKey, $params = [])
    {
        // یافتن ناحیه نمایش
        $zone = AdDisplayZone::where('zone_key', $zoneKey)
            ->where('is_active', 1)
            ->first();

        if (! $zone) {
            return collect();
        }

        // تعیین تعداد حداکثر کارت‌ها
        $limit = $zone->getMaxCardsAttribute();

        // آماده‌سازی کوئری
        $query = LocationAdPackage::query()
            ->with(['location', 'package', 'package.cardDesign'])
            ->join('locations', 'locations.id', '=', 'location_ad_packages.location_id')
            ->join('ad_packages', 'ad_packages.id', '=', 'location_ad_packages.package_id')
            ->where('location_ad_packages.is_active', 1)
            ->where('locations.is_active', 1)
            ->whereDate('location_ad_packages.end_date', '>=', now())
            ->whereIn('ad_packages.package_key', $zone->allowed_packages ?? ['diamond', 'gold', 'silver', 'bronze']);

        // اعمال فیلتر جغرافیایی
        if ($zone->is_geographic_filtered && isset($params['geo_filters'])) {
            $geoFilters = $params['geo_filters'];

            $query->where(function ($q) use ($geoFilters) {
                $q->where('location_ad_packages.geographic_scope', 'global');

                if (isset($geoFilters['province_id'])) {
                    $q->orWhere(function ($sq) use ($geoFilters) {
                        $sq->where('location_ad_packages.geographic_scope', 'province')
                            ->where('location_ad_packages.geo_id', $geoFilters['province_id']);
                    });
                }

                if (isset($geoFilters['county_id'])) {
                    $q->orWhere(function ($sq) use ($geoFilters) {
                        $sq->where('location_ad_packages.geographic_scope', 'county')
                            ->where('location_ad_packages.geo_id', $geoFilters['county_id']);
                    });
                }

                if (isset($geoFilters['city_id'])) {
                    $q->orWhere(function ($sq) use ($geoFilters) {
                        $sq->where('location_ad_packages.geographic_scope', 'city')
                            ->where('location_ad_packages.geo_id', $geoFilters['city_id']);
                    });
                }

                if (isset($geoFilters['zone_id'])) {
                    $q->orWhere(function ($sq) use ($geoFilters) {
                        $sq->where('location_ad_packages.geographic_scope', 'zone')
                            ->where('location_ad_packages.geo_id', $geoFilters['zone_id']);
                    });
                }
            });
        }

        // اعمال فیلتر دسته‌بندی
        if ($zone->is_category_filtered && isset($params['category_id'])) {
            $categoryId = $params['category_id'];

            $query->where(function ($q) use ($categoryId) {
                $q->where('location_ad_packages.category_id', $categoryId)
                    ->orWhereNull('location_ad_packages.category_id');
            });
        }

        // مرتب‌سازی
        $query->orderBy('ad_packages.priority')
            ->orderBy('location_ad_packages.display_priority')
            ->inRandomOrder();

        return $query->select('location_ad_packages.*')->limit($limit)->get();
    }

    /**
     * خرید پکیج تبلیغاتی برای یک مکان
     */
    public function purchasePackage($locationId, $packageId, $options = [])
    {
        $package = AdPackage::findOrFail($packageId);
        $location = Location::findOrFail($locationId);

        // محاسبه تاریخ پایان
        $startDate = $options['start_date'] ?? now()->format('Y-m-d');
        $endDate = $options['end_date'] ?? date('Y-m-d', strtotime($startDate." + {$package->duration_days} days"));

        // ثبت پکیج برای مکان
        $locationPackage = new LocationAdPackage([
            'location_id' => $locationId,
            'package_id' => $packageId,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'is_active' => 0, // غیرفعال تا پرداخت تکمیل شود
            'payment_status' => 'pending',
            'payment_amount' => $package->price,
            'display_priority' => $options['display_priority'] ?? 10,
            'geographic_scope' => $options['geographic_scope'] ?? null,
            'geo_id' => $options['geo_id'] ?? null,
            'category_id' => $options['category_id'] ?? null,
            'custom_badge_text' => $options['badge_text'] ?? null,
            'custom_badge_color' => $options['badge_color'] ?? null,
        ]);

        $locationPackage->save();

        return $locationPackage;
    }

    /**
     * تکمیل پرداخت پکیج تبلیغاتی
     */
    public function completePayment($locationPackageId, $paymentRef)
    {
        $locationPackage = LocationAdPackage::findOrFail($locationPackageId);

        $locationPackage->payment_status = 'paid';
        $locationPackage->payment_date = now();
        $locationPackage->payment_ref = $paymentRef;
        $locationPackage->is_active = 1;

        $locationPackage->save();

        return $locationPackage;
    }

    /**
     * دریافت تبلیغات الماسی برای صفحه اصلی
     *
     * @param  int  $limit  تعداد تبلیغات مورد نیاز
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getHomepageDiamondAds($limit = 4)
    {
        // پیدا کردن زون مربوط به تبلیغات الماسی صفحه اصلی
        $zone = DisplayZone::where('zone_key', 'homepage_diamond')
            ->where('is_active', true)
            ->first();

        if (! $zone) {
            return collect();
        }

        // دریافت تبلیغات فعال از این زون
        return ZoneAd::with(['ad', 'location'])
            ->where('zone_id', $zone->id)
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->orderBy('display_order')
            ->take($limit)
            ->get();
    }
}
