<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Amenity;
use App\Models\Category;
use App\Models\Location;
use App\Models\Rating;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show the admin dashboard.
     */
    public function index()
    {
        // Enhanced statistics with growth calculations
        $stats = $this->getEnhancedStats();

        // Recent locations with more details
        $recentLocations = Location::with(['categoryRelation', 'ratings'])
            ->withCount('ratings')
            ->withAvg('ratings', 'rating')
            ->latest()
            ->take(5)
            ->get();

        // Recent ratings with location details
        $recentRatings = Rating::with('location')
            ->latest('timestamp')
            ->take(5)
            ->get();

        // Recent advertisements with performance metrics
        $recentAds = \App\Models\Advertisement::latest()
            ->take(5)
            ->get();

        // Get latest 5 locations with enhanced data
        $latestLocations = Location::with('categoryRelation')
            ->withCount('ratings')
            ->withAvg('ratings', 'rating')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Get top rated locations
        $topRatedLocations = DB::table('ratings')
            ->select('title_id', DB::raw('AVG(rating) as average_rating'), DB::raw('COUNT(*) as ratings_count'))
            ->groupBy('title_id')
            ->orderBy('average_rating', 'desc')
            ->orderBy('ratings_count', 'desc')
            ->limit(5)
            ->get();

        // Get locations for the top rated title_ids
        $topLocations = [];
        foreach ($topRatedLocations as $rated) {
            $location = Location::with('categoryRelation')->where('title_id', $rated->title_id)->first();
            if ($location) {
                $location->average_rating = $rated->average_rating;
                $location->ratings_count = $rated->ratings_count;
                $topLocations[] = $location;
            }
        }

        // Get locations by category counts with enhanced data
        $categoryCounts = DB::table('locations')
            ->join('categories', 'locations.category_id', '=', 'categories.id')
            ->select('categories.category_fa as category', 'categories.emoji', 'categories.color', DB::raw('count(*) as count'))
            ->where('locations.is_active', 1)
            ->groupBy('categories.category_fa', 'categories.emoji', 'categories.color')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // Analytics data
        $analyticsData = $this->getAnalyticsData();

        // Monthly stats for the current year
        $monthlyStats = $this->getMonthlyStats();

        // Performance metrics
        $performanceMetrics = $this->getPerformanceMetrics();

        // Real-time metrics
        $realTimeMetrics = $this->getRealTimeMetrics();

        // Growth trends
        $growthTrends = $this->getGrowthTrends();

        return view('admin.dashboard', compact(
            'stats',
            'recentLocations',
            'recentRatings',
            'recentAds',
            'latestLocations',
            'topLocations',
            'categoryCounts',
            'analyticsData',
            'monthlyStats',
            'performanceMetrics',
            'realTimeMetrics',
            'growthTrends'
        ));
    }

    /**
     * Get analytics data for dashboard charts
     */
    private function getAnalyticsData()
    {
        // Get locations growth over time (last 12 months)
        $locationsGrowth = DB::table('locations')
            ->select(DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'), DB::raw('COUNT(*) as count'))
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Format dates for chart labels
        $months = [];
        $locationCounts = [];

        foreach ($locationsGrowth as $growth) {
            $date = Carbon::createFromFormat('Y-m', $growth->month);
            $months[] = $date->format('F Y');
            $locationCounts[] = $growth->count;
        }

        // Get ratings distribution
        $ratingsDistribution = DB::table('ratings')
            ->select('rating', DB::raw('COUNT(*) as count'))
            ->groupBy('rating')
            ->orderBy('rating')
            ->get();

        $ratingsLabels = [];
        $ratingsCounts = [];

        foreach ($ratingsDistribution as $distribution) {
            $ratingsLabels[] = $distribution->rating;
            $ratingsCounts[] = $distribution->count;
        }

        // Weekly activity
        $weeklyActivity = [
            'locations' => $this->getWeeklyActivity('locations'),
            'ratings' => $this->getWeeklyActivity('ratings', 'timestamp'),
        ];

        return [
            'months' => $months,
            'locationCounts' => $locationCounts,
            'ratingsLabels' => $ratingsLabels,
            'ratingsCounts' => $ratingsCounts,
            'weeklyActivity' => $weeklyActivity,
        ];
    }

    /**
     * Get weekly activity for a table
     */
    private function getWeeklyActivity($table, $dateField = 'created_at')
    {
        $startDate = Carbon::now()->startOfWeek();
        $endDate = Carbon::now()->endOfWeek();

        $activity = DB::table($table)
            ->select(DB::raw("DATE_FORMAT($dateField, '%w') as day_of_week"), DB::raw('COUNT(*) as count'))
            ->whereBetween($dateField, [$startDate, $endDate])
            ->groupBy('day_of_week')
            ->orderBy('day_of_week')
            ->get();

        $days = ['یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنج‌شنبه', 'جمعه', 'شنبه'];
        $counts = array_fill(0, 7, 0);

        foreach ($activity as $item) {
            $counts[$item->day_of_week] = $item->count;
        }

        return [
            'labels' => $days,
            'counts' => $counts,
        ];
    }

    /**
     * Get monthly stats for the current year
     */
    private function getMonthlyStats()
    {
        $currentYear = Carbon::now()->year;
        $startDate = Carbon::createFromDate($currentYear, 1, 1)->startOfMonth();
        $endDate = Carbon::createFromDate($currentYear, 12, 31)->endOfMonth();

        // Monthly locations added
        $monthlyLocations = DB::table('locations')
            ->select(DB::raw('MONTH(created_at) as month'), DB::raw('COUNT(*) as count'))
            ->whereYear('created_at', $currentYear)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Monthly ratings added
        $monthlyRatings = DB::table('ratings')
            ->select(DB::raw('MONTH(timestamp) as month'), DB::raw('COUNT(*) as count'))
            ->whereYear('timestamp', $currentYear)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $monthNames = [
            1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد',
            4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
            7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
            10 => 'دی', 11 => 'بهمن', 12 => 'اسفند',
        ];

        $locationData = array_fill(1, 12, 0);
        $ratingData = array_fill(1, 12, 0);

        foreach ($monthlyLocations as $item) {
            $locationData[$item->month] = $item->count;
        }

        foreach ($monthlyRatings as $item) {
            $ratingData[$item->month] = $item->count;
        }

        return [
            'labels' => array_values($monthNames),
            'locationData' => array_values($locationData),
            'ratingData' => array_values($ratingData),
        ];
    }

    /**
     * Get performance metrics
     */
    private function getPerformanceMetrics()
    {
        // Average rating score
        $avgRating = DB::table('ratings')->avg('rating') ?: 0;

        // Locations growth rate (comparing this month to previous month)
        $thisMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();

        $locationsThisMonth = Location::where('created_at', '>=', $thisMonth)->count();
        $locationsLastMonth = Location::whereBetween('created_at', [$lastMonth, $thisMonth])->count();

        $locationsGrowthRate = $locationsLastMonth > 0
            ? round((($locationsThisMonth - $locationsLastMonth) / $locationsLastMonth) * 100, 2)
            : 100;

        // Category with most locations
        $topCategory = DB::table('locations')
            ->join('categories', 'locations.category_id', '=', 'categories.id')
            ->select('categories.category_fa as category', DB::raw('count(*) as count'))
            ->groupBy('categories.category_fa')
            ->orderBy('count', 'desc')
            ->first();

        // Amenity usage stats
        $topAmenities = DB::table('location_amenities')
            ->join('amenities', 'amenities.id', '=', 'location_amenities.amenity_id')
            ->select('amenities.title', DB::raw('count(*) as count'))
            ->groupBy('amenities.id', 'amenities.title')
            ->orderBy('count', 'desc')
            ->limit(5)
            ->get();

        return [
            'avgRating' => round($avgRating, 2),
            'locationsGrowthRate' => $locationsGrowthRate,
            'locationsThisMonth' => $locationsThisMonth,
            'locationsLastMonth' => $locationsLastMonth,
            'topCategory' => $topCategory,
            'topAmenities' => $topAmenities,
        ];
    }

    /**
     * Get enhanced statistics with growth calculations
     */
    private function getEnhancedStats()
    {
        $now = Carbon::now();
        $lastMonth = $now->copy()->subMonth();
        $lastWeek = $now->copy()->subWeek();

        // Current counts
        $currentStats = [
            'locations' => Location::count(),
            'active_locations' => Location::where('is_active', 1)->count(),
            'categories' => Category::count(),
            'ratings' => Rating::count(),
            'approved_ratings' => Rating::where('status', 'approved')->count(),
            'pending_ratings' => Rating::where('status', 'pending')->count(),
            'advertisements' => \App\Models\Advertisement::count(),
            'active_ads' => \App\Models\Advertisement::where('is_active', 1)->count(),
            'admins' => Admin::count(),
        ];

        // Growth calculations
        $growthStats = [];
        foreach (['locations', 'ratings', 'advertisements'] as $entity) {
            $table = $entity === 'advertisements' ? 'advertisements' : $entity;
            $dateField = $entity === 'ratings' ? 'timestamp' : 'created_at';

            $thisMonth = DB::table($table)->where($dateField, '>=', $now->startOfMonth())->count();
            $previousMonth = DB::table($table)
                ->whereBetween($dateField, [$lastMonth->startOfMonth(), $lastMonth->endOfMonth()])
                ->count();

            $growthRate = $previousMonth > 0
                ? round((($thisMonth - $previousMonth) / $previousMonth) * 100, 1)
                : ($thisMonth > 0 ? 100 : 0);

            $growthStats[$entity . '_growth'] = $growthRate;
            $growthStats[$entity . '_this_month'] = $thisMonth;
        }

        return array_merge($currentStats, $growthStats);
    }

    /**
     * Get real-time metrics
     */
    private function getRealTimeMetrics()
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        $thisWeek = Carbon::now()->startOfWeek();
        $lastWeek = Carbon::now()->subWeek()->startOfWeek();

        return [
            'today_locations' => Location::whereDate('created_at', $today)->count(),
            'yesterday_locations' => Location::whereDate('created_at', $yesterday)->count(),
            'today_ratings' => Rating::whereDate('timestamp', $today)->count(),
            'yesterday_ratings' => Rating::whereDate('timestamp', $yesterday)->count(),
            'week_locations' => Location::where('created_at', '>=', $thisWeek)->count(),
            'last_week_locations' => Location::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
            'avg_rating_today' => Rating::whereDate('timestamp', $today)->avg('rating') ?: 0,
            'avg_rating_week' => Rating::where('timestamp', '>=', $thisWeek)->avg('rating') ?: 0,
        ];
    }

    /**
     * Get growth trends for charts
     */
    private function getGrowthTrends()
    {
        // Last 30 days trend
        $days = [];
        $locationCounts = [];
        $ratingCounts = [];

        for ($i = 29; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $days[] = $date->format('M d');

            $locationCounts[] = Location::whereDate('created_at', $date)->count();
            $ratingCounts[] = Rating::whereDate('timestamp', $date)->count();
        }

        return [
            'days' => $days,
            'locations' => $locationCounts,
            'ratings' => $ratingCounts,
        ];
    }
}
