<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Models\Rating;
use Illuminate\Http\Request;

class ReviewManagementController extends Controller
{
    /**
     * نمایش لیست نظرات برای مدیریت
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // استخراج فیلترها از درخواست
        $status = $request->input('status', 'pending');
        $locationId = $request->input('location_id');
        $search = $request->input('search');

        // ایجاد کوئری
        $query = Rating::query();

        // اعمال فیلتر وضعیت
        if ($status === 'approved') {
            $query->where('is_approved', true);
        } elseif ($status === 'pending') {
            $query->where('is_approved', false);
        }

        // اعمال فیلتر مکان اگر انتخاب شده باشد
        if ($locationId) {
            $query->where('location_id', $locationId);
        }

        // اعمال جستجو
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('positive_comment', 'like', "%{$search}%")
                    ->orWhere('negative_comment', 'like', "%{$search}%")
                    ->orWhere('comment', 'like', "%{$search}%");
            });
        }

        // مرتب‌سازی نظرات (جدیدترین ابتدا)
        $query->orderBy('created_at', 'desc');

        // دریافت نظرات با صفحه‌بندی
        $reviews = $query->with('location')->paginate(15);

        // دریافت لیست مکان‌ها برای فیلتر
        $locations = Location::orderBy('title')->get(['id', 'title']);

        return view('admin.reviews.index', [
            'reviews' => $reviews,
            'locations' => $locations,
            'status' => $status,
            'selectedLocation' => $locationId,
            'search' => $search,
        ]);
    }

    /**
     * نمایش اطلاعات یک نظر برای بررسی و مدیریت
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $review = Rating::with('location')->findOrFail($id);

        return view('admin.reviews.show', [
            'review' => $review,
        ]);
    }

    /**
     * تایید یک نظر
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve($id)
    {
        $review = Rating::findOrFail($id);
        $review->is_approved = true;
        $review->save();

        return redirect()->route('admin.reviews.index')
            ->with('success', __('نظر با موفقیت تایید شد.'));
    }

    /**
     * رد یک نظر
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reject(Request $request, $id)
    {
        $review = Rating::findOrFail($id);

        // اگر یادداشت ادمین ارسال شده باشد، آن را ذخیره می‌کنیم
        if ($request->has('admin_note')) {
            $review->admin_note = $request->admin_note;
        }

        $review->is_approved = false;
        $review->save();

        return redirect()->route('admin.reviews.index')
            ->with('success', __('نظر با موفقیت رد شد.'));
    }

    /**
     * حذف یک نظر
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $review = Rating::findOrFail($id);
        $review->delete();

        return redirect()->route('admin.reviews.index')
            ->with('success', __('نظر با موفقیت حذف شد.'));
    }

    /**
     * به‌روزرسانی یادداشت ادمین برای یک نظر
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateNote(Request $request, $id)
    {
        $request->validate([
            'admin_note' => 'nullable|string|max:500',
        ]);

        $review = Rating::findOrFail($id);
        $review->admin_note = $request->admin_note;
        $review->save();

        return redirect()->route('admin.reviews.show', $id)
            ->with('success', __('یادداشت با موفقیت به‌روزرسانی شد.'));
    }

    /**
     * نمایش تنظیمات سیستم نظرات
     *
     * @return \Illuminate\Http\Response
     */
    public function settings()
    {
        return view('admin.reviews.settings');
    }

    /**
     * ذخیره تنظیمات سیستم نظرات
     *
     * @return \Illuminate\Http\Response
     */
    public function saveSettings(Request $request)
    {
        $request->validate([
            'auto_approve' => 'boolean',
            'allow_anonymous' => 'boolean',
            'min_comment_length' => 'nullable|integer|min:0|max:500',
            'max_comment_length' => 'nullable|integer|min:0|max:5000',
        ]);

        // ذخیره تنظیمات در فایل کانفیگ یا دیتابیس
        config(['reviews.auto_approve' => (bool) $request->auto_approve]);
        config(['reviews.allow_anonymous' => (bool) $request->allow_anonymous]);
        config(['reviews.min_comment_length' => $request->min_comment_length]);
        config(['reviews.max_comment_length' => $request->max_comment_length]);

        // در حالت واقعی باید تنظیمات را در دیتابیس یا فایل ذخیره کنید

        return redirect()->route('admin.reviews.settings')
            ->with('success', __('تنظیمات با موفقیت به‌روزرسانی شدند.'));
    }
}
