<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Category;
use App\Models\Location;
use App\Models\Rating;
use App\Services\AdService;
use App\Services\DisplaySettingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;

class LocationController extends Controller
{
    protected $adService;

    protected $displaySettingService;

    public function __construct(AdService $adService, DisplaySettingService $displaySettingService)
    {
        $this->adService = $adService;
        $this->displaySettingService = $displaySettingService;
    }

    /**
     * Display a location with its proper template
     *
     * @param  string  $titleId
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        try {
            $location = Location::with(['categoryRelation.activeTemplateRelation', 'amenities'])
                ->where('is_active', 1)
                ->findOrFail($id);

            // Get related locations using the same method as showWithCategories
            $relatedLocations = $this->getRelatedLocations($location);

            // Get ratings using title_id with Rating model to get HTML formatted comments
            $ratings = \App\Models\Rating::where('title_id', $location->title_id)
                ->where('status', 'approved')
                ->orderBy('created_at', 'desc')
                ->take(10)
                ->get();

            // Get rating statistics
            $ratingStats = DB::table('rating_statistics')
                ->where('title_id', $location->title_id)
                ->first();

            // If no statistics exist, calculate from existing ratings
            if (!$ratingStats && $ratings->count() > 0) {
                $avgRating = $ratings->avg('rating');
                $totalRatings = $ratings->count();

                $ratingStats = (object) [
                    'title_id' => $location->title_id,
                    'total_ratings' => $totalRatings,
                    'average_rating' => round($avgRating, 2),
                    'service_avg' => $ratings->whereNotNull('service_rating')->avg('service_rating'),
                    'quality_avg' => $ratings->whereNotNull('quality_rating')->avg('quality_rating'),
                    'location_avg' => $ratings->whereNotNull('location_rating')->avg('location_rating'),
                    'price_avg' => $ratings->whereNotNull('price_rating')->avg('price_rating'),
                    'cleanliness_avg' => $ratings->whereNotNull('cleanliness_rating')->avg('cleanliness_rating'),
                ];
            }

            // Empty category hierarchy for consistency
            $categoryHierarchy = [];

            return view('locations.show', compact('location', 'relatedLocations', 'ratings', 'ratingStats', 'categoryHierarchy'));
        } catch (\Exception $e) {
            \Log::error('Error in LocationController@show: '.$e->getMessage());
            abort(404);
        }
    }

    /**
     * Display a location with category hierarchy in URL
     *
     * @param  string  $titleId
     * @return \Illuminate\View\View
     */
    public function showWithCategories(Request $request, $categoryPath, $titleId)
    {
        try {
            // $categoryPath and $titleId are now directly passed from the route.
            // $titleId is already validated by the route constraint '.*-\d+$'
            // $categoryPath contains the category segments.
            $pathParts = explode('/', $categoryPath);

            // Ensure titleId is not null or empty, though route constraint should handle this.
            if (empty($titleId)) {
                Log::error('titleId is empty in showWithCategories after route processing.');
                abort(404, 'Location identifier is missing.');
            }

            // Find the location with its category
            $location = Location::with([
                'province',
                'county',
                'city',
                'locationAmenities.amenity',
                'seo',
                'ratings',
                'categoryRelation' => function($query) {
                    $query->with(['parent' => function($query) {
                        $query->with('parent');
                    }, 'activeTemplateRelation']);
                }
            ])->where('title_id', $titleId)
                ->where('is_active', true)
                ->firstOrFail();

            // Get the category hierarchy
            $categoryHierarchy = $this->getCategoryHierarchyFromLocation($location);

            // Extract category keys from URL (which is $categoryPath)
            $urlCategoryKeys = $pathParts;

            // Generate correct category keys based on the location's categories
            $correctCategoryKeys = [];
            foreach ($categoryHierarchy as $category) {
                if (is_object($category) && isset($category->category_key) && ! empty($category->category_key)) {
                    $correctCategoryKeys[] = $category->category_key;
                }
            }

            // Log the current state for debugging
            \Log::info('Category comparison', [
                'url_categories' => $urlCategoryKeys,
                'correct_categories' => $correctCategoryKeys,
                'current_url' => url()->current(),
                'location_id' => $location->id,
                'category_id' => $location->category_id,
                'category_type' => is_object($location->category) ? get_class($location->category) : gettype($location->category)
            ]);

            // Special handling for shopping centers and malls
            if ($location->category_id) {
                $category = Category::find($location->category_id);
                if ($category) {
                    if ($category->category_key === 'mall') {
                        $correctCategoryKeys = ['shopping_centers', 'mall'];
                    } elseif ($category->category_key === 'shopping_centers') {
                        $correctCategoryKeys = ['shopping_centers'];
                    }
                }
            }

            // Check if we need to redirect
            $needsRedirect = false;
            if (!empty($correctCategoryKeys)) {
                // Remove 'fa' from URL categories if it exists
                $urlCategoryKeys = array_filter($urlCategoryKeys, function($key) {
                    return $key !== 'fa';
                });

                // Normalize arrays for comparison
                $normalizedUrlKeys = array_map('strtolower', array_values($urlCategoryKeys));
                $normalizedCorrectKeys = array_map('strtolower', $correctCategoryKeys);

                // Check if arrays are different
                if ($normalizedUrlKeys !== $normalizedCorrectKeys) {
                    $needsRedirect = true;
                }
            }

            // Only redirect if necessary and if the URL is different
            if ($needsRedirect) {
                // Construct the correct path using the $correctCategoryKeys and $titleId
                $correctPath = implode('/', $correctCategoryKeys) . '/' . $titleId;
                $correctUrl = url($correctPath);

                // Double check that we're not redirecting to the same URL
                // $request->path() gives the path without the domain, e.g., 'shopping_centers/mall/bazar-myaaad-laft-1745931891'
                // We need to compare the path part of the URL
                $currentPathForComparison = $categoryPath . '/' . $titleId;
                if ($correctPath !== $currentPathForComparison && $correctUrl !== url()->current()) {
                    \Log::info('Redirecting to correct URL', [
                        'from' => url()->current(),
                        'to' => $correctUrl,
                        'url_keys' => $normalizedUrlKeys,
                        'correct_keys' => $normalizedCorrectKeys
                    ]);
                    return redirect($correctUrl, 301);
                }
            }

            // Get template based on category hierarchy or fall back to standard method if empty
            $templateView = !empty($categoryHierarchy) ?
                $this->getTemplateViewFromHierarchy($categoryHierarchy, $location) :
                $this->getTemplateView($location);

            // Get ratings using title_id with Rating model to get HTML formatted comments
            $ratings = \App\Models\Rating::where('title_id', $location->title_id)
                ->where('status', 'approved')
                ->orderBy('created_at', 'desc')
                ->take(10)
                ->get();

            // Get rating statistics
            $ratingStats = DB::table('rating_statistics')
                ->where('title_id', $location->title_id)
                ->first();

            // If no statistics exist, calculate from existing ratings
            if (!$ratingStats && $ratings->count() > 0) {
                $avgRating = $ratings->avg('rating');
                $totalRatings = $ratings->count();

                $ratingStats = (object) [
                    'title_id' => $location->title_id,
                    'total_ratings' => $totalRatings,
                    'average_rating' => round($avgRating, 2),
                    'service_avg' => $ratings->whereNotNull('service_rating')->avg('service_rating'),
                    'quality_avg' => $ratings->whereNotNull('quality_rating')->avg('quality_rating'),
                    'location_avg' => $ratings->whereNotNull('location_rating')->avg('location_rating'),
                    'price_avg' => $ratings->whereNotNull('price_rating')->avg('price_rating'),
                    'cleanliness_avg' => $ratings->whereNotNull('cleanliness_rating')->avg('cleanliness_rating'),
                ];
            }

            // Get related locations
            $relatedLocations = $this->getRelatedLocations($location);

            // Set empty categoryHierarchy if needed to avoid blade template errors
            if (empty($categoryHierarchy)) {
                $categoryHierarchy = [];
            }

            // Log final template selection
            \Log::info('Using template', [
                'template' => $templateView,
                'location_id' => $location->id,
                'related_count' => $relatedLocations->count()
            ]);

            return view($templateView, compact('location', 'ratings', 'ratingStats', 'categoryHierarchy', 'relatedLocations'));

        } catch (\Exception $e) {
            \Log::error('Error in showWithCategories: ' . $e->getMessage(), [
                'exception' => $e,
                'url' => $request->fullUrl()
            ]);
            abort(404);
        }
    }

    /**
     * Get category hierarchy array from a location
     *
     * @return array
     */
    protected function getCategoryHierarchyFromLocation(Location $location)
    {
        $hierarchy = [];

        if (! $location->category_id) {
            return $hierarchy;
        }

        // Start with the immediate category
        $category = $location->category;

        // Check if category is actually an object and not a string
        if (! is_object($category)) {
            // Try to load category from database if $location->category_id exists
            $category = Category::find($location->category_id);
            if (! $category) {
                return $hierarchy;
            }
        }

        // Special handling for shopping centers
        if ($category->category_key === 'shopping_centers') {
            return [$category];
        }

        // We'll build the hierarchy from child to parent (bottom-up)
        $categoryChain = [];
        $currentCategory = $category;

        // اطمینان از بارگذاری ترجمه‌ها و فیلدهای لازم برای دسته فعلی
        if ($currentCategory && !isset($currentCategory->translations)) {
            $currentCategory = Category::with('translations')->find($currentCategory->id);
        }
        $categoryChain[] = $currentCategory;

        // Then recursively add parents
        while ($currentCategory && isset($currentCategory->parent_id) && $currentCategory->parent_id > 0) {
            // بارگذاری والد با ترجمه‌ها و فیلدهای لازم
            $parentCategory = Category::with('translations')->find($currentCategory->parent_id);
            if ($parentCategory) {
                $categoryChain[] = $parentCategory;
                $currentCategory = $parentCategory;
            } else {
                break;
            }
        }

        // Reverse the order to get parent->child ordering (top-down)
        return array_reverse($categoryChain);
    }

    /**
     * Get template view based on category hierarchy
     *
     * @return string
     */
    protected function getTemplateViewFromHierarchy(array $hierarchy, Location $location)
    {
        // Default template
        $defaultTemplate = 'templates.locations.base';

        // If no hierarchy, use standard method
        if (empty($hierarchy)) {
            return $this->getTemplateView($location);
        }

        // First check if there's a direct template for the location type
        if ($location->type) {
            $directTemplate = 'templates.locations.'.$location->type;
            if (View::exists($directTemplate)) {
                \Log::info("Using direct type template: {$directTemplate}");
                return $directTemplate;
            }
        }

        // Try to find a template for the most specific (last) category first,
        // then work up the hierarchy to more general templates
        for ($i = count($hierarchy) - 1; $i >= 0; $i--) {
            $category = $hierarchy[$i];

            // Skip invalid category entries
            if (! is_object($category)) {
                continue;
            }

            // First check the specific depth-based template
            $levelName = $this->getCategoryLevelName($i, count($hierarchy));

            // Check if category has category_key property
            if (property_exists($category, 'category_key') && $category->category_key) {
                // Try child template first
                $childTemplate = 'templates.locations.child.'.$category->category_key;
                if (View::exists($childTemplate)) {
                    \Log::info("Using child template: {$childTemplate}");
                    return $childTemplate;
                }

                // Then try hierarchy template
                $hierarchyTemplate = 'templates.locations.'.$levelName.'.'.$category->category_key;
                if (View::exists($hierarchyTemplate)) {
                    \Log::info("Using hierarchy template: {$hierarchyTemplate}");
                    return $hierarchyTemplate;
                }
            }

            // Then check for a template using this category directly
            try {
                // Try to get template directly from the category
                if (method_exists($category, 'getTemplateViewName')) {
                    $templateView = $category->getTemplateViewName();
                    if (View::exists($templateView)) {
                        \Log::info("Using template from category method: {$templateView}");
                        return $templateView;
                    }
                }

                // Try with category key
                if (property_exists($category, 'category_key') && $category->category_key) {
                    $categoryKeyTemplate = 'templates.locations.'.$category->category_key;
                    if (View::exists($categoryKeyTemplate)) {
                        \Log::info("Using category key template: {$categoryKeyTemplate}");
                        return $categoryKeyTemplate;
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error getting template view: '.$e->getMessage());
            }
        }

        // If no specific template found, use the default
        \Log::info("Using default template: {$defaultTemplate}");
        return $defaultTemplate;
    }

    /**
     * Get a descriptive name for the category level
     *
     * @param  int  $index  Current index in the hierarchy array
     * @param  int  $total  Total number of items in hierarchy
     * @return string
     */
    protected function getCategoryLevelName($index, $total)
    {
        // For a hierarchy like [grandparent, parent, child], index 0 is the grandparent
        if ($index == 0 && $total > 1) {
            return 'grandparent';
        } elseif (($index == 0 && $total == 1) || ($index == 1 && $total > 2)) {
            return 'parent';
        } elseif (($index == 1 && $total == 2) || ($index == 2 && $total > 2)) {
            return 'child';
        } else {
            return 'grandchild';
        }
    }

    /**
     * Get the appropriate template view for a location
     *
     * @return string
     */
    protected function getTemplateView(Location $location)
    {
        // Default template
        $defaultTemplate = 'templates.locations.base';

        // Check if we have a category object
        if (isset($location->categoryObj)) {
            return $this->getTemplateFromCategory($location->categoryObj);
        }

        // If no category object available, try to use the category_id
        if ($location->category_id) {
            $category = Category::find($location->category_id);
            if ($category) {
                return $this->getTemplateFromCategory($category);
            }
        }

        // If we have a category string, try to find template based on that
        if (is_string($location->category) && ! empty($location->category)) {
            // Try to use category string as template key
            $templateKey = strtolower(str_replace(' ', '_', $location->category));
            $categoryTemplate = 'templates.locations.'.$templateKey;

            if (View::exists($categoryTemplate)) {
                return $categoryTemplate;
            }
        }

        // Default fallback
        return $defaultTemplate;
    }

    /**
     * Get template from a category object
     *
     * @return string
     */
    protected function getTemplateFromCategory(Category $category)
    {
        // Try to get the template view name from the category
        try {
            $templateView = $category->getTemplateViewName();

            // Check if view exists
            if (View::exists($templateView)) {
                return $templateView;
            }

            // Use category key as fallback
            if ($category->category_key) {
                $categoryKeyTemplate = 'templates.locations.'.$category->category_key;
                if (View::exists($categoryKeyTemplate)) {
                    return $categoryKeyTemplate;
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting template view: '.$e->getMessage());
        }

        // Default fallback
        return 'templates.locations.base';
    }

    /**
     * Get related locations for a given location
     *
     * @param Location $location
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getRelatedLocations(Location $location)
    {
        // Start with locations from the same category
        $relatedQuery = Location::where('is_active', true)
            ->where('id', '!=', $location->id)
            ->with(['categoryRelation']);

        // First try: Same category
        $sameCategory = (clone $relatedQuery)
            ->where('category_id', $location->category_id)
            ->take(4)
            ->get();

        // If we have enough from same category, return them
        if ($sameCategory->count() >= 3) {
            return $sameCategory->take(3);
        }

        // If not enough, try same city
        $sameCity = (clone $relatedQuery)
            ->where('city_id', $location->city_id)
            ->whereNotIn('id', $sameCategory->pluck('id'))
            ->take(3 - $sameCategory->count())
            ->get();

        $related = $sameCategory->merge($sameCity);

        // If still not enough, try same province
        if ($related->count() < 3) {
            $sameProvince = (clone $relatedQuery)
                ->where('province_id', $location->province_id)
                ->whereNotIn('id', $related->pluck('id'))
                ->take(3 - $related->count())
                ->get();

            $related = $related->merge($sameProvince);
        }

        // If still not enough, get any recent locations
        if ($related->count() < 3) {
            $recent = (clone $relatedQuery)
                ->whereNotIn('id', $related->pluck('id'))
                ->orderBy('created_at', 'desc')
                ->take(3 - $related->count())
                ->get();

            $related = $related->merge($recent);
        }

        return $related->take(3);
    }

    /**
     * Generate a category path URL for a location
     *
     * @return string
     */
    public function generateCategoryPath(Location $location)
    {
        // Default path if no category info is available
        $path = "/l/{$location->title_id}";

        // If location has a category, use the category path
        if ($location->category_id) {
            $category = Category::find($location->category_id);
            if ($category) {
                $categoryParts = $this->getCategoryHierarchy($category);
                if (! empty($categoryParts)) {
                    $path = '/'.implode('/', $categoryParts).'/'.$location->title_id;
                }
            }
        }

        return $path;
    }

    /**
     * Get the full category hierarchy as an array of slugs
     *
     * @return array
     */
    protected function getCategoryHierarchy(Category $category)
    {
        $parts = [];
        $current = $category;

        // Add current category
        $slug = $this->generateSlug($current->category_fa);
        $parts[] = $slug;

        // Add parent categories recursively
        while ($current->parent_id) {
            $current = Category::find($current->parent_id);
            if ($current) {
                $slug = $this->generateSlug($current->category_fa);
                array_unshift($parts, $slug);
            } else {
                break;
            }
        }

        return $parts;
    }

    /**
     * Generate a URL-friendly slug from a string
     *
     * @param  string  $text
     * @return string
     */
    protected function generateSlug($text)
    {
        // Convert to lowercase
        $text = mb_strtolower($text, 'UTF-8');

        // Replace spaces with hyphens
        $text = str_replace(' ', '-', $text);

        // Remove any character that's not a letter, number, hyphen or underscore
        $text = preg_replace('/[^\p{L}\p{N}_-]/u', '', $text);

        // Remove multiple hyphens
        $text = preg_replace('/-+/', '-', $text);

        // Trim hyphens from the beginning and end
        return trim($text, '-');
    }

    /**
     * Increment the view count for a location
     *
     * @return void
     */
    protected function incrementViewCount(Location $location)
    {
        try {
            // Update view count in a separate query to prevent blocking
            dispatch(function () use ($location) {
                $location->increment('views_count');
            })->afterResponse();
        } catch (\Exception $e) {
            // Log error but don't interrupt page loading
            \Log::error('Error incrementing view count: '.$e->getMessage());
        }
    }

    /**
     * نمایش لیست مکان‌ها در صفحه اصلی
     */
    public function index(Request $request)
    {
        $isMobile = $request->isMobile();
        $perPage = $this->displaySettingService->getItemsPerPage('locations', 'list', $isMobile);

        $query = Location::query()
            ->where('is_active', 1)
            ->with(['categoryRelation.activeTemplateRelation', 'city', 'province'])
            ->orderBy('created_at', 'desc');

        // اعمال فیلترها
        if ($request->has('category')) {
            $query->where('category_id', $request->category);
        }

        if ($request->has('city')) {
            $query->where('city_id', $request->city);
        }

        if ($request->has('province')) {
            $query->where('province_id', $request->province);
        }

        $locations = $query->paginate($perPage);

        if ($request->ajax()) {
            return response()->json([
                'html' => view('partials.location-list', compact('locations'))->render(),
                'hasMorePages' => $locations->hasMorePages(),
            ]);
        }

        $loadMoreText = $this->displaySettingService->getLoadMoreText('locations', 'list');

        return view('locations.index', compact('locations', 'loadMoreText'));
    }

    public function category($categorySlug)
    {
        $category = \App\Models\Category::where('slug', $categorySlug)->firstOrFail();

        // دریافت تبلیغات ویژه دسته‌بندی
        $categoryAds = $this->adService->getAdsForZone('category_featured', [
            'category_id' => $category->id,
        ]);

        // دریافت تبلیغات پایین صفحه دسته‌بندی
        $bottomAds = $this->adService->getAdsForZone('category_bottom', [
            'category_id' => $category->id,
        ]);

        return view('category', compact('category', 'categoryAds', 'bottomAds'));
    }

    /**
     * نمایش نتایج جستجو بر اساس منطقه جغرافیایی
     */
    public function geoSearch(Request $request)
    {
        $geoFilters = [
            'province_id' => $request->input('province_id'),
            'county_id' => $request->input('county_id'),
            'city_id' => $request->input('city_id'),
            'zone_id' => $request->input('zone_id'),
        ];

        // دریافت تبلیغات منطقه جغرافیایی
        $geoAds = $this->adService->getAdsForZone('geo_featured', [
            'geo_filters' => $geoFilters,
        ]);

        // دریافت نتایج جستجو
        $searchResults = \App\Models\Location::query()
            ->when($request->filled('province_id'), function ($q) use ($request) {
                $q->where('province_id', $request->input('province_id'));
            })
            ->when($request->filled('county_id'), function ($q) use ($request) {
                $q->where('county_id', $request->input('county_id'));
            })
            ->when($request->filled('city_id'), function ($q) use ($request) {
                $q->where('city_id', $request->input('city_id'));
            })
            ->when($request->filled('zone_id'), function ($q) use ($request) {
                $q->where('zone_id', $request->input('zone_id'));
            })
            ->paginate(20);

        return view('search.geo', compact('geoAds', 'searchResults', 'geoFilters'));
    }
}
