<?php

namespace App\Models;

use App\Traits\HasTranslatedName;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Province extends Model
{
    use HasFactory, SoftDeletes, HasTranslatedName;

    protected $table = 'state2_provinces';

    protected $fillable = [
        'country_id',
        'name',
        'name_en',
        'code',
        'slug',
        'global_slug',
        'center_city_name',
        'boundaries',
        'area',
        'population',
        'description',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'area' => 'decimal:2',
        'population' => 'integer',
        'deleted_at' => 'datetime',
    ];

    /**
     * Get the boundaries data in properly formatted GeoJSON
     */
    public function getBoundariesAttribute($value)
    {
        // Log the raw value for debugging
        \Log::debug('Raw boundaries value: '.substr(json_encode($value), 0, 100).'...');

        if (! $value) {
            \Log::debug('Province boundaries is empty or null');

            return null;
        }

        // Check if this might be binary or corrupted data
        if (! is_string($value)) {
            \Log::debug('Province boundaries is not a string');

            return is_array($value) || is_object($value) ? json_encode($value) : (string) $value;
        }

        if (! $this->isValidUTF8($value)) {
            \Log::debug('Province boundaries is not valid UTF-8');
            // Try to convert it to UTF-8
            $converted = mb_convert_encoding($value, 'UTF-8', 'UTF-8');

            return $converted;
        }

        if ($this->containsBinaryData($value)) {
            \Log::debug('Province boundaries contains binary data');
            // Clean the binary data but don't return null
            $cleaned = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x9F]/u', '', $value);

            return $cleaned;
        }

        // Try to sanitize
        $sanitized = preg_replace('/[\x00-\x1F\x7F]/u', '', $value);

        // Verify it's valid JSON, but don't return null if it's not
        if ($this->isValidJSON($sanitized)) {
            return $sanitized;
        }

        \Log::debug('Returning original value after failed validation');

        // Return the raw value as a fallback
        return $value;
    }

    /**
     * Check if a string is valid UTF-8
     */
    protected function isValidUTF8($string)
    {
        return mb_check_encoding($string, 'UTF-8');
    }

    /**
     * Check if a string appears to contain binary data
     */
    protected function containsBinaryData($string)
    {
        // Check for common binary file signatures
        $binarySignatures = [
            "\x1F\x8B", // gzip
            "\x42\x5A", // bzip2
            "\x50\x4B", // zip
            "\x89\x50\x4E\x47", // png
            "\xFF\xD8\xFF", // jpg
            '%PDF', // pdf
            'PK', // zip/docx/xlsx
            'BM', // bmp
            "\x7FELF", // ELF binary
            "\xD0\xCF\x11\xE0", // MS Office
        ];

        foreach ($binarySignatures as $signature) {
            if (strpos($string, $signature) === 0) {
                return true;
            }
        }

        // Check for high concentration of non-printable characters
        $nonPrintable = preg_match_all('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x9F]/', $string);
        if ($nonPrintable > 0 && ($nonPrintable / strlen($string)) > 0.1) {
            return true;
        }

        return preg_match('/[\x00-\x08\x0B\x0C\x0E-\x1F]/', $string) > 0;
    }

    /**
     * Check if a string is valid JSON
     */
    protected function isValidJSON($string)
    {
        json_decode($string);

        return json_last_error() == JSON_ERROR_NONE;
    }

    /**
     * Set the boundaries value for proper storage
     */
    public function setBoundariesAttribute($value)
    {
        if (! $value) {
            $this->attributes['boundaries'] = null;

            return;
        }

        // If value is a WKT string (for MySQL GEOMETRY type)
        if (is_string($value) && (
            strpos($value, 'POLYGON') === 0 ||
            strpos($value, 'MULTIPOLYGON') === 0 ||
            strpos($value, 'LINESTRING') === 0
        )) {
            // Store as is for GEOMETRY columns
            $this->attributes['boundaries'] = $value;

            return;
        }

        // For JSON storage
        try {
            // If it's already a string but not WKT, check if it's valid JSON
            if (is_string($value)) {
                json_decode($value);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $this->attributes['boundaries'] = $value;

                    return;
                }
            }

            // If it's an array or object, encode it
            if (is_array($value) || is_object($value)) {
                $this->attributes['boundaries'] = json_encode($value);

                return;
            }

            // Default case
            $this->attributes['boundaries'] = (string) $value;
        } catch (\Exception $e) {
            \Log::error('Error setting boundaries: '.$e->getMessage());
            $this->attributes['boundaries'] = null;
        }
    }

    /**
     * دریافت نام ترجمه‌شده استان بر اساس زبان فعال
     * 
     * @param string $languageCode کد زبان (fa, en, ...)
     * @return string نام ترجمه‌شده
     */
    public function getTranslatedName($lang = null)
    {
        $lang = $lang ?: app()->getLocale();
        if ($lang === 'en' && !empty($this->name_en)) {
            return $this->name_en;
        }
        return $this->name;
    }

    // Relationships
    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function counties()
    {
        return $this->hasMany(County::class);
    }

    public function locations()
    {
        return $this->hasMany(Location::class);
    }
}
