<?php

namespace App\Providers;

use App\Models\Advertisement;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Facades\Blade;
use App\Services\SiteSettingService;
use App\Helpers\SiteSettings;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        $this->app->singleton('site-settings', function ($app) {
            return new SiteSettings();
        });

        $this->app->singleton(SiteSettingService::class, function ($app) {
            return new SiteSettingService();
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Register jdate helper function for Persian date formatting
        if (! function_exists('jdate')) {
            function jdate($date = null, $format = null)
            {
                if (is_null($date)) {
                    return Jalalian::now();
                }

                if ($date instanceof \Carbon\Carbon) {
                    return Jalalian::fromCarbon($date);
                }

                if (is_numeric($date)) {
                    return Jalalian::fromFormat('Y-m-d H:i:s', date('Y-m-d H:i:s', $date));
                }

                return Jalalian::forge($date)->format($format);
            }
        }

        // Register Blade directives for date formatting
        \Blade::directive('dateByLocale', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::formatByLocale($expression); ?>";
        });

        \Blade::directive('dateTimeByLocale', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::formatDateTimeByLocale($expression); ?>";
        });

        \Blade::directive('dateForDisplay', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::formatForDisplay($expression); ?>";
        });

        \Blade::directive('dateRelative', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::formatRelativeByLocale($expression); ?>";
        });

        \Blade::directive('dateContext', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::formatForContext($expression); ?>";
        });

        \Blade::directive('persianNumbers', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::convertToPersianNumbers($expression); ?>";
        });

        \Blade::directive('englishNumbers', function ($expression) {
            return "<?php echo \App\Helpers\DateHelper::convertToEnglishNumbers($expression); ?>";
        });

        // Get advertisements for homepage by position
        View::composer('*', function ($view) {
            $adPositions = [
                'homepage_hero',
                'homepage_sidebar',
                'homepage_featured',
                'homepage_middle',
                'homepage_bottom',
                'popup',
            ];

            $adsByPosition = [];

            foreach ($adPositions as $position) {
                $adsByPosition[$position] = Advertisement::active()
                    ->byPosition($position)
                    ->orderBy('display_order')
                    ->limit(5)
                    ->get();
            }

            $view->with('adsByPosition', $adsByPosition);
        });

        Blade::directive('site_setting', function ($key) {
            return "<?php echo site_setting($key); ?>";
        });
    }
}
